# Version 3.7 - Service-Agnostic Payment Processing System with Order Type Fix
# Author: Code Omaha  
# Studio: Omaha Affiliates

import os
import imaplib
import email
import re
import requests
import logging
import sqlite3
from dotenv import load_dotenv
from bs4 import BeautifulSoup
from datetime import datetime, timedelta
import json

# ===== This is the payment push notification block =====
def send_payment_received_notification(telegram_user_id, order_id, amount, bot_ref):
    try:
        msg = (
            f"💳 <b>Payment Received!</b>\n\n"
            f"Order #{order_id}\n"
            f"Payment Reference: <code>{bot_ref}</code>\n"
            f"Amount: ${amount:.2f}\n\n"
            "We're now processing your order. You'll receive another message when it's ready!"
        )
        send_telegram_notification(telegram_user_id, msg)
    except Exception as e:
        logging.error(f"Failed to notify payment receipt: {e}")

def send_order_completed_notification(telegram_user_id, order_id, service_name, username=None):
    try:
        msg = (
            f"✅ <b>Your order #{order_id} is complete!</b>\n\n"
            f"Service: {service_name}\n"
        )
        if username:
            msg += f"Account: <code>{username}</code>\n"
        msg += "\nThank you for your payment and trust!\n\n💬 Need help? Contact support."
        send_telegram_notification(telegram_user_id, msg)
    except Exception as e:
        logging.error(f"Failed to notify order completion: {e}")
# ===== This is the end of payment push notification block =====

# =========== CONFIGURATION AND LOGGING ===========

load_dotenv()

IMAP_SERVER = os.environ.get('IMAP_SERVER')
IMAP_FOLDER = os.environ.get('IMAP_FOLDER', 'INBOX')
EMAIL_ACCOUNT = os.environ.get('EMAIL_ACCOUNT')
EMAIL_PASSWORD = os.environ.get('EMAIL_PASSWORD')
PROCESSED_LOG = os.environ.get('PROCESSED_LOG', 'processed_emails.log')

# Bot integration settings
BOT_DB_PATH = "bot_data.db"
BOT_TOKEN = os.environ.get('BOT_TOKEN')

logging.basicConfig(
    level=logging.INFO, 
    format='%(asctime)s %(levelname)s:%(message)s',
    handlers=[
        logging.FileHandler("payment_processing.log", encoding='utf-8'),
        logging.StreamHandler()
    ]
)

# =========== BOT INTEGRATION CLASSES ===========

class BotDatabase:
    def __init__(self, db_path: str = BOT_DB_PATH):
        self.db_path = db_path
        self.migrate_account_links()
    
    def migrate_account_links(self):
        """Add service_name column to account_links if it doesn't exist"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        
        try:
            cursor.execute("PRAGMA table_info(account_links)")
            columns = [column[1] for column in cursor.fetchall()]
            
            if 'service_name' not in columns:
                cursor.execute("ALTER TABLE account_links ADD COLUMN service_name TEXT")
                logging.info("Added service_name column to account_links table")
            
            conn.commit()
        except Exception as e:
            logging.error(f"Migration error: {e}")
        finally:
            conn.close()
    
    def get_order_by_bot_reference(self, bot_ref: str):
        """Get specific order by Bot payment reference"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute("""
            SELECT id, telegram_user_id, service_id, order_type, package_id, panel_package_id, 
                   username, password, total_price, xxx_addon, service_name, package_name, created_at, telegram_username
            FROM orders 
            WHERE notes = ? AND status = 'pending_payment' AND payment_status = 'pending'
        """, (bot_ref,))
        order = cursor.fetchone()
        conn.close()
        return order
    
    def get_all_pending_orders(self):
        """Get all pending orders for debugging"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute("""
            SELECT id, telegram_user_id, total_price, status, payment_status, created_at, notes
            FROM orders 
            WHERE status = 'pending_payment' AND payment_status = 'pending'
            ORDER BY created_at DESC
        """)
        orders = cursor.fetchall()
        conn.close()
        return orders
    
    def fulfill_order(self, order_id: int, username: str, password: str, notes: str = ""):
        """Mark order as fulfilled"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute("""
            UPDATE orders 
            SET status = 'completed', payment_status = 'paid', username = ?, password = ?, 
                notes = ?, fulfilled_at = CURRENT_TIMESTAMP, updated_at = CURRENT_TIMESTAMP 
            WHERE id = ?
        """, (username, password, notes, order_id))
        affected_rows = cursor.rowcount
        conn.commit()
        conn.close()
        logging.info(f"Database update: {affected_rows} rows affected for order #{order_id}")
        return affected_rows > 0

    def add_wallet_funds(self, user_id: int, amount: float, description: str, order_id: int = None) -> bool:
        """Add funds to user's wallet"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        
        try:
            # Get current balance
            cursor.execute("SELECT balance FROM wallet WHERE user_id = ?", (user_id,))
            result = cursor.fetchone()
            current_balance = result[0] if result else 0.0
            new_balance = current_balance + amount
            
            # Update or insert wallet balance
            cursor.execute("""
                INSERT OR REPLACE INTO wallet (user_id, balance, updated_at) 
                VALUES (?, ?, CURRENT_TIMESTAMP)
            """, (user_id, new_balance))
            
            # Record transaction
            cursor.execute("""
                INSERT INTO wallet_transactions 
                (user_id, transaction_type, amount, balance_before, balance_after, description, order_id)
                VALUES (?, 'credit', ?, ?, ?, ?, ?)
            """, (user_id, amount, current_balance, new_balance, description, order_id))
            
            conn.commit()
            return True
        except Exception as e:
            conn.rollback()
            logging.error(f"Error adding wallet funds: {e}")
            return False
        finally:
            conn.close()

    def get_wallet_balance(self, user_id: int) -> float:
        """Get user's wallet balance"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        cursor.execute("SELECT balance FROM wallet WHERE user_id = ?", (user_id,))
        result = cursor.fetchone()
        conn.close()
        return result[0] if result else 0.0

    def auto_link_account(self, telegram_user_id: int, telegram_username: str, panel_username: str, order_id: int, service_name: str):
        """Automatically link a completed order account to the user with service name"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        
        try:
            # Check if link already exists
            cursor.execute("""
                SELECT id FROM account_links 
                WHERE telegram_user_id = ? AND panel_username = ?
            """, (telegram_user_id, panel_username))
            
            existing_link = cursor.fetchone()
            
            if not existing_link:
                # Create automatic link with approved status and service name
                cursor.execute("""
                    INSERT INTO account_links (telegram_user_id, panel_username, service_name, status, verification_code, linked_at, approved_by, created_at)
                    VALUES (?, ?, ?, 'approved', ?, CURRENT_TIMESTAMP, 0, CURRENT_TIMESTAMP)
                """, (telegram_user_id, panel_username, service_name, f"AUTO_ORDER_{order_id}"))
                
                conn.commit()
                logging.info(f"AUTO-LINK: Account {panel_username} ({service_name}) automatically linked to user {telegram_user_id} (@{telegram_username}) from order #{order_id}")
                return True
            else:
                logging.info(f"AUTO-LINK: Account {panel_username} already linked to user {telegram_user_id}")
                return False
                
        except Exception as e:
            logging.error(f"AUTO-LINK: Error linking account {panel_username} to user {telegram_user_id}: {e}")
            conn.rollback()
            return False
        finally:
            conn.close()

    def fix_existing_accounts(self):
        """Fix existing accounts that are labeled incorrectly"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        
        try:
            # Get all orders that have accounts linked but may have wrong service name
            cursor.execute("""
                SELECT DISTINCT o.username, o.service_name, al.id, al.service_name as current_service
                FROM orders o
                JOIN account_links al ON o.username = al.panel_username
                WHERE o.service_name != al.service_name AND o.status = 'completed'
            """)
            
            accounts_to_fix = cursor.fetchall()
            
            for username, correct_service, link_id, current_service in accounts_to_fix:
                cursor.execute("""
                    UPDATE account_links 
                    SET service_name = ? 
                    WHERE id = ?
                """, (correct_service, link_id))
                logging.info(f"FIXED: Updated account {username} service from {current_service} to {correct_service}")
            
            conn.commit()
            logging.info(f"Fixed {len(accounts_to_fix)} accounts with incorrect service labels")
            
        except Exception as e:
            logging.error(f"Error fixing account service labels: {e}")
            conn.rollback()
        finally:
            conn.close()

class PanelManager:
    def __init__(self):
        self.panels = {}  # Cache for panel instances
    
    def get_panel_for_service(self, service_name: str):
        """Get the appropriate panel API instance for a service using environment variables"""
        if service_name in self.panels:
            return self.panels[service_name]
        
        # Search through environment variables to find matching service
        panel_config = None
        
        for service_num in range(1, 6):  # Check SERVICE1 through SERVICE5
            env_service_name = os.getenv(f"SERVICE{service_num}_NAME")
            if env_service_name and env_service_name == service_name:
                panel_config = {
                    "url": os.getenv(f"SERVICE{service_num}_PANEL_URL"),
                    "username": os.getenv(f"SERVICE{service_num}_PANEL_USERNAME"),
                    "password": os.getenv(f"SERVICE{service_num}_PANEL_PASSWORD")
                }
                break
        
        if not panel_config:
            logging.error(f"No configuration found for service: {service_name}")
            return None
        
        # Validate config - ensure all required values are present
        if not all([panel_config["url"], panel_config["username"], panel_config["password"]]):
            logging.error(f"Missing panel configuration for {service_name}")
            logging.error(f"URL: {panel_config['url']}, Username: {panel_config['username']}, Password: {'***' if panel_config['password'] else 'None'}")
            return None
        
        # Create and cache panel instance
        panel = XtreamPanelAPI(panel_config["url"], panel_config["username"], panel_config["password"])
        self.panels[service_name] = panel
        
        logging.info(f"PANEL: Configured for {service_name}: {panel_config['url']}")
        return panel

class XtreamPanelAPI:
    def __init__(self, panel_url: str, username: str, password: str):
        self.panel_url = panel_url.rstrip('/')
        self.username = username
        self.password = password
        self.session = requests.Session()
        
    def authenticate(self):
        """Authenticate with the Xtream UI panel"""
        self.session = requests.Session()
        
        try:
            # Set headers to mimic browser
            self.session.headers.update({
                'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
                'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
                'Accept-Language': 'en-US,en;q=0.5',
                'Accept-Encoding': 'gzip, deflate',
                'Connection': 'keep-alive',
                'Upgrade-Insecure-Requests': '1'
            })
            
            logging.info(f"PANEL: Connecting to {self.panel_url}")
            
            # Get login page
            login_page = self.session.get(f"{self.panel_url}/login.php", timeout=10)
            if login_page.status_code != 200:
                return {"success": False, "message": f"Cannot access panel: {login_page.status_code}"}
            
            # Submit login
            login_data = {
                "username": self.username,
                "password": self.password
            }
            
            login_response = self.session.post(
                f"{self.panel_url}/login.php", 
                data=login_data, 
                timeout=10, 
                allow_redirects=True
            )
            
            # Check for successful login
            if login_response.status_code == 200 and (
                "dashboard" in login_response.url.lower() or 
                "admin" in login_response.url.lower() or
                "logout" in login_response.text.lower() or
                "index.php" in login_response.url or
                "reseller.php" in login_response.url
            ):
                logging.info(f"PANEL: Authentication successful - redirected to {login_response.url}")
                return {"success": True, "message": "Authentication successful", "final_url": login_response.url}
            else:
                logging.error(f"PANEL: Authentication failed")
                return {"success": False, "message": "Authentication failed"}
                
        except Exception as e:
            logging.error(f"PANEL: Authentication error: {str(e)}")
            return {"success": False, "message": f"Authentication error: {str(e)}"}
    
    def search_user_get_id(self, username: str):
        """Search for user and return their internal ID and details"""
        try:
            logging.info(f"PANEL: Searching for user {username} to get ID")
            
            # Use table search to find the user
            params = {
                'draw': 1,
                'start': 0,
                'length': 250,
                'search[value]': username.lower(),
                'search[regex]': 'false',
                'id': 'users',
                'filter': '',
                'reseller': '',
                '_': str(int(datetime.now().timestamp() * 1000))
            }
            
            for i in range(12):
                params.update({
                    f'columns[{i}][data]': str(i),
                    f'columns[{i}][name]': '',
                    f'columns[{i}][searchable]': 'true',
                    f'columns[{i}][orderable]': 'true' if i != 11 else 'false',
                    f'columns[{i}][search][value]': '',
                    f'columns[{i}][search][regex]': 'false'
                })
            
            params.update({
                'order[0][column]': '0',
                'order[0][dir]': 'desc'
            })
            
            search_url = f"{self.panel_url}/table_search.php"
            response = self.session.get(search_url, params=params, timeout=15)
            
            if response.status_code == 200:
                try:
                    data = response.json()
                    if isinstance(data, dict) and 'data' in data:
                        for record in data['data']:
                            if len(record) >= 8 and record[1].lower() == username.lower():
                                user_id = record[0]  # Internal ID
                                current_expiry = record[7]  # Current expiration
                                logging.info(f"PANEL: Found user {username} with ID {user_id}, current expiry: {current_expiry}")
                                return {
                                    "found": True,
                                    "user_id": user_id,
                                    "username": record[1],
                                    "password": record[2],
                                    "current_expiry": current_expiry,
                                    "status": record[4],
                                    "max_connections": record[9] if len(record) > 9 else "1"
                                }
                except json.JSONDecodeError:
                    logging.error("PANEL: Search response not valid JSON")
            
            logging.error(f"PANEL: User {username} not found in search results")
            return {"found": False}
            
        except Exception as e:
            logging.error(f"PANEL: Error searching for user {username}: {e}")
            return {"found": False, "error": str(e)}
    
    def renew_user_account(self, username: str, package_id: str, duration_days: int, notes: str = ""):
        """Renew existing account using the proper panel API"""
        
        # Authenticate first
        auth_result = self.authenticate()
        if not auth_result["success"]:
            return {"error": f"Authentication failed: {auth_result['message']}"}
        
        try:
            logging.info(f"PANEL: Starting renewal for account {username}")
            logging.info(f"  Package ID: {package_id}")
            logging.info(f"  Duration: {duration_days} days")
            
            # Step 1: Find the user and get their ID
            user_search = self.search_user_get_id(username)
            if not user_search.get("found"):
                return {"error": f"User {username} not found in panel"}
            
            user_id = user_search["user_id"]
            current_expiry = user_search["current_expiry"]
            
            logging.info(f"PANEL: Found user {username} with ID {user_id}")
            logging.info(f"PANEL: Current expiry: {current_expiry}")
            
            # Step 2: Use the panel's API to get extension details
            api_url = f"{self.panel_url}/api.php"
            api_params = {
                'action': 'get_package',
                'package_id': package_id,
                'user_id': user_id
            }
            
            logging.info(f"PANEL: Getting package extension details from API")
            api_response = self.session.get(api_url, params=api_params, timeout=10)
            
            if api_response.status_code == 200:
                try:
                    api_data = api_response.json()
                    if api_data.get('result') is True:
                        package_data = api_data['data']
                        new_expiry = package_data.get('exp_date')
                        cost_credits = package_data.get('cost_credits', 0)
                        max_connections = package_data.get('max_connections', 1)
                        
                        logging.info(f"PANEL: API returned package extension data:")
                        logging.info(f"  New expiry: {new_expiry}")
                        logging.info(f"  Cost: {cost_credits} credits")
                        logging.info(f"  Max connections: {max_connections}")
                        
                        # Step 3: Submit the renewal form
                        edit_url = f"{self.panel_url}/user_reseller.php?id={user_id}"
                        
                        # Create form data for the renewal
                        renewal_form_data = {
                            'edit': user_id,
                            'username': user_search["username"],
                            'password': user_search["password"],
                            'member_id': '42',
                            'package': package_id,
                            'max_connections': str(max_connections),
                            'exp_date': new_expiry,
                            'reseller_notes': notes,
                            'submit_user': 'Purchase'
                        }
                        
                        logging.info(f"PANEL: Submitting renewal form: {renewal_form_data}")
                        
                        # Submit the renewal
                        renewal_response = self.session.post(
                            edit_url,
                            data=renewal_form_data,
                            timeout=15,
                            allow_redirects=True
                        )
                        
                        if renewal_response.status_code == 200:
                            logging.info(f"PANEL: Renewal submitted successfully")
                            
                            # Step 4: Verify the renewal worked
                            import time
                            time.sleep(3)  # Wait for processing
                            
                            verify_search = self.search_user_get_id(username)
                            if verify_search.get("found"):
                                updated_expiry = verify_search["current_expiry"]
                                logging.info(f"PANEL: Verification - Updated expiry: {updated_expiry}")
                                
                                # Check if renewal was successful
                                if updated_expiry != current_expiry:
                                    logging.info(f"PANEL: RENEWAL SUCCESS - Expiry updated from {current_expiry} to {updated_expiry}")
                                    return {
                                        "success": True,
                                        "username": username,
                                        "method": "api_renewal",
                                        "duration_added": duration_days,
                                        "package_id": package_id,
                                        "old_expiry": current_expiry,
                                        "new_expiry": updated_expiry,
                                        "verified": True,
                                        "cost_credits": cost_credits
                                    }
                                else:
                                    # Account still exists even if expiry didn't change
                                    return {
                                        "success": True,
                                        "username": username,
                                        "method": "api_renewal",
                                        "duration_added": duration_days,
                                        "package_id": package_id,
                                        "old_expiry": current_expiry,
                                        "new_expiry": new_expiry,
                                        "verified": False,
                                        "warning": "Account preserved but expiry verification unclear"
                                    }
                            else:
                                return {"error": "Account was deleted during renewal"}
                        else:
                            return {"error": f"Renewal form submission failed: HTTP {renewal_response.status_code}"}
                    else:
                        return {"error": f"API returned error: {api_data}"}
                except json.JSONDecodeError:
                    return {"error": "API response is not valid JSON"}
            else:
                return {"error": f"API call failed: HTTP {api_response.status_code}"}
                
        except Exception as e:
            logging.error(f"PANEL: Renewal error for {username}: {str(e)}")
            return {"error": f"Renewal error: {str(e)}"}
    
    def get_package_details(self, package_id):
        """Get package details via API call"""
        try:
            logging.info(f"PANEL: Getting package details for ID {package_id}")
            
            # Call the API that the JavaScript uses
            api_url = f"{self.panel_url}/api.php?action=get_package&package_id={package_id}"
            response = self.session.get(api_url, timeout=10)
            
            if response.status_code == 200:
                try:
                    data = response.json()
                    if data.get('result') is True:
                        logging.info(f"PANEL: Package details received: {data['data']}")
                        return data['data']
                    else:
                        logging.error(f"PANEL: API returned false result: {data}")
                except:
                    logging.error(f"PANEL: API response not JSON: {response.text[:200]}")
            else:
                logging.error(f"PANEL: API call failed: {response.status_code}")
                
        except Exception as e:
            logging.error(f"PANEL: Error getting package details: {e}")
        
        return None
    
    def extract_credentials_from_edit_page(self, html_content):
        """Extract username and password from edit user page"""
        try:
            soup = BeautifulSoup(html_content, 'html.parser')
            
            # Find username field
            username_input = soup.find('input', {'name': 'username'})
            username = username_input.get('value', '') if username_input else ''
            
            # Find password field
            password_input = soup.find('input', {'name': 'password'})
            password = password_input.get('value', '') if password_input else ''
            
            logging.info(f"PANEL: Extracted credentials - Username: {username}, Password: {password}")
            
            return username, password
            
        except Exception as e:
            logging.error(f"PANEL: Error extracting credentials: {e}")
            return None, None
    
    def verify_account_exists(self, user_id):
        """Verify the account still exists after creation"""
        try:
            logging.info(f"PANEL: Verifying account exists - User ID: {user_id}")
            
            # Try to access the edit page again
            verify_url = f"{self.panel_url}/user_reseller.php?id={user_id}"
            response = self.session.get(verify_url, timeout=10)
            
            if response.status_code == 200:
                if "edit user" in response.text.lower() and user_id in response.text:
                    logging.info(f"PANEL: Account verification PASSED - User {user_id} still exists")
                    return True
                else:
                    logging.error(f"PANEL: Account verification FAILED - User {user_id} may have been deleted")
                    return False
            else:
                logging.error(f"PANEL: Account verification error - HTTP {response.status_code}")
                return False
                
        except Exception as e:
            logging.error(f"PANEL: Account verification exception: {e}")
            return False
    
    def create_user_account(self, package_id: str, duration_days: int, max_connections: int = 1, notes: str = ""):
        """Create account with enhanced debugging and proper verification"""
        
        # Authenticate first
        auth_result = self.authenticate()
        if not auth_result["success"]:
            return {"error": f"Authentication failed: {auth_result['message']}"}
        
        try:
            logging.info(f"PANEL: Creating account with auto-generated credentials:")
            logging.info(f"  Package ID: {package_id}")
            logging.info(f"  Duration: {duration_days} days")
            logging.info(f"  Max Connections: {max_connections}")
            
            # Step 1: Get package details via API
            package_details = self.get_package_details(package_id)
            if not package_details:
                logging.error("PANEL: Could not get package details")
                return {"error": "Could not get package details from API"}
            
            # Extract details from API response
            api_max_connections = package_details.get('max_connections', max_connections)
            api_exp_date = package_details.get('exp_date', '')
            cost_credits = package_details.get('cost_credits', 0)
            
            logging.info(f"PANEL: API returned - Max Connections: {api_max_connections}, Exp Date: {api_exp_date}, Cost: {cost_credits}")
            
            # Step 2: Submit form WITHOUT device fields
            form_data = {
                'member_id': '42',
                'package': str(package_id),
                'max_connections': str(api_max_connections),
                'exp_date': api_exp_date,
                'reseller_notes': notes,
                'submit_user': 'Purchase'
            }
            
            logging.info(f"PANEL: Submitting form data: {form_data}")
            
            # Submit to user_reseller.php
            response = self.session.post(
                f"{self.panel_url}/user_reseller.php",
                data=form_data,
                timeout=15,
                allow_redirects=True
            )
            
            logging.info(f"PANEL: Form submission response: {response.status_code}")
            logging.info(f"PANEL: Response URL: {response.url}")
            
            if response.status_code == 200:
                response_text = response.text
                
                # Check for success by looking for edit page redirect
                if "user_reseller.php?id=" in response.url:
                    # Extract user ID
                    user_id_match = re.search(r'user_reseller\.php\?id=(\d+)', response.url)
                    if user_id_match:
                        new_user_id = user_id_match.group(1)
                        logging.info(f"PANEL: Account created successfully! User ID: {new_user_id}")
                        
                        # Extract credentials
                        username, password = self.extract_credentials_from_edit_page(response_text)
                        
                        if username and password:
                            logging.info(f"PANEL: Credentials extracted: {username} / {password}")
                            
                            # Verify account exists
                            import time
                            time.sleep(2)  # Wait 2 seconds
                            
                            still_exists = self.verify_account_exists(new_user_id)
                            
                            if still_exists:
                                logging.info("PANEL: Account verification PASSED")
                                
                                return {
                                    "success": True,
                                    "username": username,
                                    "password": password,
                                    "user_id": new_user_id,
                                    "package_id": package_id,
                                    "expiry_days": duration_days,
                                    "max_connections": api_max_connections,
                                    "method": "panel_auto_generated",
                                    "verified": True
                                }
                            else:
                                logging.error("PANEL: Account completely disappeared!")
                                return {"error": "Account was created but disappeared"}
                        else:
                            logging.error("PANEL: Could not extract credentials from edit page")
                            return {"error": "Account created but could not extract credentials"}
                    else:
                        logging.error("PANEL: Could not extract user ID from redirect URL")
                        return {"error": "Account created but could not extract user ID"}
                
                # Check for explicit errors
                error_indicators = [
                    'there was an error performing this operation',
                    'please check the form entry',
                    'insufficient credits',
                    'error performing'
                ]
                
                response_lower = response_text.lower()
                found_errors = [err for err in error_indicators if err in response_lower]
                
                if found_errors:
                    logging.error(f"PANEL: Account creation failed - errors: {found_errors}")
                    return {"error": f"Panel returned errors: {found_errors}"}
                else:
                    logging.warning(f"PANEL: No redirect to edit page - unclear result")
                    return {"error": "No redirect to edit page - account may not have been created"}
            else:
                logging.error(f"PANEL: Form submission failed with status {response.status_code}")
                return {"error": f"Form submission failed: HTTP {response.status_code}"}
                
        except Exception as e:
            logging.error(f"PANEL: Account creation exception: {str(e)}")
            return {"error": f"Account creation error: {str(e)}"}

# =========== HELPER FUNCTIONS ===========

def extract_invoices_and_amount(subject):
    """Extracts Bot payment references and amount from the subject."""
    if not subject:
        return [], [], None

    # Normalize subject for consistent spacing
    subject = subject.replace('\u00A0', ' ')  # Replace non-breaking spaces
    subject = re.sub(r'\s+', ' ', subject)

    # Match Bot payment references (case-insensitive, allow optional space)
    # Accept variations like 'bot 1234', 'BOT1234', 'BoOt 1234', etc.
    bot_refs = set()
    for match in re.findall(r'bot[\s]*([0-9]{4})', subject, re.IGNORECASE):
        bot_refs.add(f"Bot{match}")

    # Match dollar amounts
    amount = None
    for pat in [
        r'\$[\s]*([\d,.]+)',
        r'sent[\s]+\$?([\d,.]+)',
        r'([\d,.]+)[\s]*dollars?',
        r'amount[\s]*:?[\s]*\$?([\d,.]+)',
    ]:
        amount_match = re.search(pat, subject, re.IGNORECASE)
        if amount_match:
            amount = amount_match.group(1).replace(',', '')
            break

    return [], list(bot_refs), amount

def get_cycle_duration_days(cycle_name):
    """Convert cycle name to duration in days"""
    cycle_map = {
        "30 Days": 30,
        "90 Days": 90,
        "180 Days": 180,
        "1 Month": 30,
        "3 Months": 90,
        "6 Months": 180,
        "30days": 30,
        "90days": 90,
        "180days": 180
    }
    return cycle_map.get(cycle_name, 30)

def send_telegram_notification(user_id: int, message: str, reply_markup: dict = None):
    """Send notification to user via Telegram bot"""
    try:
        url = f"https://api.telegram.org/bot{BOT_TOKEN}/sendMessage"
        data = {
            'chat_id': user_id,
            'text': message,
            'parse_mode': 'HTML'
        }
        
        if reply_markup:
            import json
            data['reply_markup'] = json.dumps(reply_markup)
        
        response = requests.post(url, data=data, timeout=10)
        success = response.status_code == 200
        if not success:
            logging.error(f"Telegram error: {response.text}")
        return success
    except Exception as e:
        logging.error(f"Failed to send Telegram notification: {e}")
        return False

def get_message_id(msg):
    return msg.get('Message-ID')

def load_processed_ids(log_path):
    if not os.path.exists(log_path):
        return set()
    processed = set()
    with open(log_path, 'r') as f:
        for line in f:
            line = line.strip()
            if line:
                msgid = line.split(' ')[0]
                processed.add(msgid)
    return processed

def log_processed_id(log_path, msgid, invoice_numbers=None, bot_refs=None, status="processed"):
    identifiers = []
    if invoice_numbers:
        identifiers.extend([f"INV {inv}" for inv in invoice_numbers])
    if bot_refs:
        identifiers.extend([f"BOT {bot_ref}" for bot_ref in bot_refs])
    
    id_str = ""
    if identifiers:
        id_str = " [" + ", ".join(identifiers) + f"] {status}"
    
    with open(log_path, 'a') as f:
        f.write(f"{msgid}{id_str}\n")

def mark_as_seen(mail, num):
    mail.store(num, '+FLAGS', '\\Seen')

# =========== MAIN LOGIC ===========

def main():
    required_envs = [
        IMAP_SERVER, EMAIL_ACCOUNT, EMAIL_PASSWORD, BOT_TOKEN
    ]
    if not all(required_envs):
        logging.error("Missing required environment variables.")
        return

    # Initialize
    bot_db = BotDatabase()
    panel_manager = PanelManager()
    
    # Fix existing accounts that are incorrectly labeled
    bot_db.fix_existing_accounts()
    
    processed_ids = load_processed_ids(PROCESSED_LOG)
    bot_orders_processed = []

    # Show pending orders
    pending_orders = bot_db.get_all_pending_orders()
    logging.info(f"Found {len(pending_orders)} pending orders:")
    for order in pending_orders:
        bot_ref = order[6] if len(order) > 6 else "No Bot Ref"
        logging.info(f"   Order #{order[0]}: ${order[2]:.2f} - {order[3]}/{order[4]} - {bot_ref}")

    try:
        mail = imaplib.IMAP4_SSL(IMAP_SERVER)
        mail.login(EMAIL_ACCOUNT, EMAIL_PASSWORD)
        mail.select(IMAP_FOLDER)

        # Get unread emails only
        result, data = mail.search(None, '(UNSEEN)')
        if result != 'OK':
            logging.error("Failed to fetch emails.")
            return

        email_nums = data[0].split()
        logging.info(f"Found {len(email_nums)} unread emails")

        for num in email_nums:
            result, msg_data = mail.fetch(num, '(RFC822)')
            if result != 'OK':
                continue

            msg = email.message_from_bytes(msg_data[0][1])
            msgid = get_message_id(msg)
            subject = msg['subject'] or ""
            sender = msg['from'] or ""

            # 1. Only process emails from cash@square.com
            from email.utils import parseaddr
            real_email = parseaddr(sender)[1].lower().strip()
            if real_email != 'cash@square.com':
                mark_as_seen(mail, num)
                continue

            # 2. Prevent duplicate processing
            if not msgid or msgid in processed_ids:
                mark_as_seen(mail, num)
                continue

            # 3. Require both $amount and Bot#### in subject
            _, bot_refs, amount = extract_invoices_and_amount(subject)
            if not (amount and bot_refs):
                logging.warning(f"PAYMENT: Email missing required info (bot ref or amount): {subject}")
                mark_as_seen(mail, num)
                continue

            try:
                total_amount = float(amount)
            except Exception:
                logging.error(f"PAYMENT: Invalid amount format: {amount}")
                mark_as_seen(mail, num)
                continue

            # 4. Process each Bot reference
            for bot_ref in bot_refs:
                logging.info(f"PAYMENT: Looking for order with Bot reference: {bot_ref}")

                order = bot_db.get_order_by_bot_reference(bot_ref)

                if order is None:
                    logging.warning(f"PAYMENT: No pending order found for Bot reference: {bot_ref}")
                    continue

                order_id = order[0]
                telegram_user_id = order[1]
                total_price = order[8]
                telegram_username = order[13] if len(order) > 13 and order[13] else "unknown"

                # UNDERPAYMENT: Do NOT process, notify user, log and continue
                if total_amount < total_price - 0.01:
                    diff = total_price - total_amount
                    notify_text = (
                        f"❌ <b>Payment Amount Too Low</b>\n\n"
                        f"<b>Order #{order_id}</b>\n"
                        f"<b>Reference:</b> <code>{bot_ref}</code>\n"
                        f"<b>Amount Received:</b> ${total_amount:.2f}\n"
                        f"<b>Amount Required:</b> ${total_price:.2f}\n"
                        f"<b>Shortage:</b> ${diff:.2f}\n\n"
                        "Your payment was less than required. The order has NOT been processed.\n\n"
                        "Please contact support to resolve this. Do NOT send another payment for this order!\n"
                    )
                    send_telegram_notification(telegram_user_id, notify_text)
                    log_processed_id(PROCESSED_LOG, msgid, [], [bot_ref], "PAYMENT_TOO_SHORT")
                    processed_ids.add(msgid)
                    mark_as_seen(mail, num)
                    continue

                # OVERPAYMENT: Notify user, process order as normal
                if total_amount > total_price + 0.01:
                    diff = total_amount - total_price
                    notify_text = (
                        f"⚠️ <b>Payment Amount Higher Than Required</b>\n\n"
                        f"<b>Order #{order_id}</b>\n"
                        f"<b>Reference:</b> <code>{bot_ref}</code>\n"
                        f"<b>Amount Received:</b> ${total_amount:.2f}\n"
                        f"<b>Amount Required:</b> ${total_price:.2f}\n"
                        f"<b>Overpayment:</b> ${diff:.2f}\n\n"
                        "Your payment was more than required. Your order will be processed as normal.\n\n"
                        "Please contact support if you wish to receive a refund or credit for the excess amount."
                    )
                    send_telegram_notification(telegram_user_id, notify_text)
                    # fall through to processing block (below)

                # Extract order details
                order_id = order[0]
                telegram_user_id = order[1]
                service_id = order[2]
                order_type_from_db = order[3]
                package_id = order[4]
                panel_package_id = order[5]
                username_from_order = order[6]
                password_from_order = order[7]
                total_price = order[8]
                xxx_addon = order[9]
                service_name = order[10]
                package_name = order[11]
                created_at = order[12]
                telegram_username = order[13] if len(order) > 13 and order[13] else "unknown"

                # CHECK FOR ADD FUNDS ORDER
                if order_type_from_db == "add_funds" or service_name == "Add Funds":
                    logging.info(f"ADD FUNDS: Processing wallet credit for order #{order_id}")
                    
                    # Add funds to wallet
                    if bot_db.add_wallet_funds(telegram_user_id, total_price, f"Add Funds - Order #{order_id}", order_id):
                        # Update order status
                        fulfilled = bot_db.fulfill_order(
                            order_id,
                            "WALLET_CREDIT",
                            "N/A",
                            f"Wallet credited ${total_price:.2f}. Payment: ${total_amount}. Bot Ref: {bot_ref}"
                        )
                        
                        if fulfilled:
                            # Get new balance
                            new_balance = bot_db.get_wallet_balance(telegram_user_id)
                            
                            # Send notification
                            notification = f"""💰 <b>Wallet Funds Added!</b>

<b>Order #{order_id} Completed</b>
<b>Payment Reference:</b> <code>{bot_ref}</code>

<b>Amount Added:</b> ${total_price:.2f}
<b>New Balance:</b> ${new_balance:.2f}

Your wallet has been credited successfully!
You can now use these funds for any service orders or renewals.

💬 Need help? Contact our support team."""
                            
                            main_menu_button = {
                                "inline_keyboard": [
                                    [{"text": "💰 View Wallet", "callback_data": "wallet"}],
                                    [{"text": "🏠 Main Menu", "callback_data": "back_to_start"}]
                                ]
                            }
                            
                            if send_telegram_notification(telegram_user_id, notification, main_menu_button):
                                logging.info(f"SUCCESS: Add Funds order #{order_id} completed! Added ${total_price:.2f} to wallet")
                                
                                bot_orders_processed.append({
                                    'order_id': order_id,
                                    'bot_ref': bot_ref,
                                    'username': 'WALLET_CREDIT',
                                    'user_id': 'N/A',
                                    'service': 'Add Funds',
                                    'amount': total_amount,
                                    'verified': True,
                                    'auto_linked': False,
                                    'is_renewal': False,
                                    'old_expiry': 'N/A',
                                    'new_expiry': 'N/A'
                                })
                                
                                # Mark as processed
                                log_processed_id(PROCESSED_LOG, msgid, [], [bot_ref], "ADD_FUNDS_COMPLETED")
                                processed_ids.add(msgid)
                            else:
                                logging.error(f"ERROR: Failed to send notification for Add Funds order #{order_id}")
                        else:
                            logging.error(f"ERROR: Failed to update database for Add Funds order #{order_id}")
                    else:
                        logging.error(f"ERROR: Failed to credit wallet for order #{order_id}")
                    
                    continue  # Skip panel processing for Add Funds

                # CONTINUE WITH NORMAL ORDER PROCESSING FOR NON-ADD-FUNDS ORDERS
                billing_cycle = package_name
                duration_days = get_cycle_duration_days(billing_cycle)

                # Get the correct panel for this service
                panel_api = panel_manager.get_panel_for_service(service_name)
                if not panel_api:
                    logging.error(f"ERROR: No panel configured for service {service_name}")
                    continue

                max_connections = 1
                if 'shared' in package_name.lower() or '3' in package_name:
                    max_connections = 3
                elif 'extra' in package_name.lower() or '5' in package_name:
                    max_connections = 5
                elif '2' in package_name:
                    max_connections = 2
                elif '4' in package_name:
                    max_connections = 4

                notes = f"Bot Order #{order_id} ({bot_ref}). XXX: {'Yes' if xxx_addon else 'No'} @{telegram_username if telegram_username != 'unknown' else 'UnknownUser'}"
                is_renewal = order_type_from_db == "renewal"
                existing_username = username_from_order if is_renewal and username_from_order and username_from_order.strip() else None

                # Handle bulk renewals specially
                if order_type_from_db == "bulk_renewal":
                    # Parse the account list from details
                    import json
                    try:
                        accounts_to_renew = json.loads(order[16])  # details field
                    except:
                        logging.error(f"ERROR: Could not parse bulk renewal details for order #{order_id}")
                        continue
                    
                    all_successful = True
                    renewed_accounts = []
                    failed_accounts = []
                    
                    logging.info(f"BULK RENEWAL: Processing {len(accounts_to_renew)} accounts for order #{order_id}")
                    
                    for account_info in accounts_to_renew:
                        acc_username = account_info['username']
                        acc_service_name = account_info['service_name']
                        acc_panel_package_id = account_info.get('panel_package_id', '')
                        
                        # Get panel for this service
                        panel_api = panel_manager.get_panel_for_service(acc_service_name)
                        if panel_api:
                            # Process renewal for this account
                            account_result = panel_api.renew_user_account(
                                username=acc_username,
                                package_id=acc_panel_package_id,
                                duration_days=30,
                                notes=f"Bulk renewal order #{order_id} ({bot_ref})"
                            )
                            
                            if account_result.get('success'):
                                renewed_accounts.append({
                                    'username': acc_username,
                                    'service': acc_service_name,
                                    'old_expiry': account_result.get('old_expiry', 'N/A'),
                                    'new_expiry': account_result.get('new_expiry', 'N/A')
                                })
                                logging.info(f"BULK RENEWAL: Successfully renewed {acc_username} ({acc_service_name})")
                            else:
                                failed_accounts.append({
                                    'username': acc_username,
                                    'service': acc_service_name,
                                    'error': account_result.get('error', 'Unknown error')
                                })
                                all_successful = False
                                logging.error(f"BULK RENEWAL: Failed to renew {acc_username}: {account_result.get('error')}")
                        else:
                            failed_accounts.append({
                                'username': acc_username,
                                'service': acc_service_name,
                                'error': 'No panel configured'
                            })
                            all_successful = False
                    
                    # Update order status
                    renewal_notes = f"Bulk renewal completed. Renewed: {len(renewed_accounts)}/{len(accounts_to_renew)}. Payment: ${total_amount}. Bot Ref: {bot_ref}"
                    fulfilled = bot_db.fulfill_order(order_id, "BULK", "RENEWAL", renewal_notes)
                    
                    if fulfilled:
                        # Send notification
                        if all_successful:
                            notification = f"""🎉 <b>Bulk Renewal Complete!</b>

<b>Order #{order_id}</b>
<b>Payment Reference:</b> <code>{bot_ref}</code>

✅ <b>All {len(renewed_accounts)} accounts renewed successfully!</b>

<b>Renewed Accounts:</b>
"""
                            for acc in renewed_accounts:
                                notification += f"\n• <b>{acc['username']}</b> ({acc['service']})"
                                notification += f"\n  Extended to: {acc['new_expiry']}"
                            
                            notification += "\n\n✅ All accounts extended by 30 days!"
                            notification += "\n\n💬 Need help? Contact our support team."
                        else:
                            notification = f"""⚠️ <b>Bulk Renewal Partially Complete</b>

<b>Order #{order_id}</b>
<b>Payment Reference:</b> <code>{bot_ref}</code>

✅ <b>Renewed: {len(renewed_accounts)} accounts</b>
❌ <b>Failed: {len(failed_accounts)} accounts</b>

<b>Successfully Renewed:</b>
"""
                            for acc in renewed_accounts:
                                notification += f"\n• {acc['username']} ({acc['service']})"
                            
                            if failed_accounts:
                                notification += "\n\n<b>Failed Renewals:</b>"
                                for acc in failed_accounts:
                                    notification += f"\n• {acc['username']} ({acc['service']}): {acc['error']}"
                            
                            notification += "\n\n⚠️ Please contact support for failed renewals."
                        
                        # Send notification
                        main_menu_button = {
                            "inline_keyboard": [
                                [{"text": "🏠 Main Menu", "callback_data": "back_to_start"}],
                                [{"text": "📱 My Accounts", "callback_data": "my_accounts"}]
                            ]
                        }
                        
                        send_telegram_notification(telegram_user_id, notification, main_menu_button)
                        
                        # Log and mark as processed
                        status = "BULK_RENEWAL_COMPLETED"
                        log_processed_id(PROCESSED_LOG, msgid, [], [bot_ref], status)
                        processed_ids.add(msgid)
                        logging.info(f"SUCCESS: Bulk renewal order #{order_id} completed! Renewed {len(renewed_accounts)}/{len(accounts_to_renew)} accounts")
                    
                    # Skip to next email/bot reference
                    continue

                # Check if this is a renewal or new account
                if is_renewal and existing_username:
                    # This is a renewal
                    logging.info(f"RENEWAL: Processing renewal for existing account {existing_username}")
                    notes += f" - RENEWAL - Extended by {duration_days} days"
                    
                    account_result = panel_api.renew_user_account(
                        username=existing_username,
                        package_id=panel_package_id,
                        duration_days=duration_days,
                        notes=notes
                    )
                    
                    # For renewals, use existing username and keep existing password
                    if account_result.get('success'):
                        account_result['username'] = existing_username
                        account_result['password'] = "unchanged"  # Keep existing password
                        account_result['user_id'] = "renewal"
                        account_result['max_connections'] = max_connections
                        account_result['verified'] = account_result.get('verified', True)
                else:
                    # Create new account
                    logging.info(f"NEW ACCOUNT: Creating new account for order #{order_id}")
                    account_result = panel_api.create_user_account(
                        package_id=panel_package_id,
                        duration_days=duration_days,
                        max_connections=max_connections,
                        notes=notes
                    )
                
                if account_result.get('success'):
                    username = account_result['username']
                    password = account_result['password']
                    user_id = account_result.get('user_id', 'unknown')
                    verified = account_result.get('verified', False)
                    old_expiry = account_result.get('old_expiry', 'N/A')
                    new_expiry = account_result.get('new_expiry', 'N/A')
                    cost_credits = account_result.get('cost_credits', 0)
                    
                    if is_renewal and existing_username:
                        logging.info(f"SUCCESS: Account renewed successfully!")
                        logging.info(f"  Username: {username}")
                        logging.info(f"  Duration added: {duration_days} days")
                        logging.info(f"  Old expiry: {old_expiry}")
                        logging.info(f"  New expiry: {new_expiry}")
                        logging.info(f"  Verified: {verified}")
                        logging.info(f"  Credits used: {cost_credits}")
                    else:
                        logging.info(f"SUCCESS: Account created and verified!")
                        logging.info(f"  Username: {username}")
                        logging.info(f"  User ID: {user_id}")
                    
                    # Update database with Bot reference preserved
                    renewal_notes = f"{'Renewal' if is_renewal else 'Account created'} (ID: {user_id}). Payment: ${total_amount}. Bot Ref: {bot_ref}"
                    if is_renewal and existing_username:
                        renewal_notes += f". Extended from {old_expiry} to {new_expiry}. Verified: {verified}. Credits: {cost_credits}"
                    
                    fulfilled = bot_db.fulfill_order(
                        order_id, 
                        username, 
                        password, 
                        renewal_notes
                    )
                    
                    if fulfilled:
                        # AUTO-LINK: Automatically link the account to the user (only for new accounts)
                        auto_link_success = False
                        if not (is_renewal and existing_username):  # Only auto-link new accounts
                            auto_link_success = bot_db.auto_link_account(
                                telegram_user_id=telegram_user_id,
                                telegram_username=telegram_username,
                                panel_username=username,
                                order_id=order_id,
                                service_name=service_name
                            )
                        
                        # Send notification with working credentials
                        if is_renewal and existing_username:
                            # Renewal notification with expiry details
                            notification = f"""🔄 <b>Account Renewed!</b>

<b>Order #{order_id} Completed</b>
<b>Payment Reference:</b> <code>{bot_ref}</code>

<b>Renewed Account:</b> <code>{username}</code>
<b>Service:</b> {service_name}
<b>Package:</b> {package_name}
<b>XXX Content:</b> {'Included' if xxx_addon else 'Not Included'}

<b>📋 Renewal Details:</b>
<b>Extension:</b> {duration_days} days added
<b>Previous Expiry:</b> {old_expiry}
<b>New Expiry:</b> {new_expiry}
<b>Connections:</b> {account_result['max_connections']}
<b>Password:</b> Unchanged

Your account has been successfully extended!

✅ <b>Account is already linked to your Telegram!</b>
You can view updated details in "My Accounts" menu.

💬 Need help? Contact our support team."""
                        else:
                            # New account notification
                            notification = f"""🎉 <b>Account Ready!</b>

<b>Order #{order_id} Completed</b>
<b>Payment Reference:</b> <code>{bot_ref}</code>

<b>Service:</b> {service_name}
<b>Package:</b> {package_name}
<b>XXX Content:</b> {'Included' if xxx_addon else 'Not Included'}

<b>🔐 Login Details:</b>
<b>Username:</b> <code>{username}</code>
<b>Password:</b> <code>{password}</code>
<b>Connections:</b> {account_result['max_connections']}
<b>Valid Until:</b> {duration_days} days from now

Your account is now active and ready to use!

✅ <b>Account automatically linked to your Telegram!</b>
You can now view this account in "My Accounts" menu.

💬 Need help? Contact our support team."""
                        
                        # Create main menu button
                        if is_renewal and existing_username:
                            # For renewals, don't show copy buttons (they already know the credentials)
                            main_menu_button = {
                                "inline_keyboard": [
                                    [{"text": "🏠 Main Menu", "callback_data": "back_to_start"}],
                                    [{"text": "📱 My Accounts", "callback_data": "my_accounts"}]
                                ]
                            }
                        else:
                            # For new accounts, show copy buttons
                            main_menu_button = {
                                "inline_keyboard": [
                                    [{"text": "📋 Copy Username", "callback_data": f"copy_user_{username}"},
                                     {"text": "📋 Copy Password", "callback_data": f"copy_pass_{username}"}],
                                    [{"text": "🏠 Main Menu", "callback_data": "back_to_start"}],
                                    [{"text": "📱 My Accounts", "callback_data": "my_accounts"}]
                                ]
                            }
                        
                        if send_telegram_notification(telegram_user_id, notification, main_menu_button):
                            logging.info(f"SUCCESS: Order #{order_id} FULLY COMPLETED!")
                            if auto_link_success:
                                logging.info(f"SUCCESS: Account {username} ({service_name}) automatically linked to @{telegram_username}")
                            elif is_renewal and existing_username:
                                logging.info(f"SUCCESS: Account {username} renewed for @{telegram_username} - Extended to {new_expiry}")
                            
                            bot_orders_processed.append({
                                'order_id': order_id,
                                'bot_ref': bot_ref,
                                'username': username,
                                'user_id': user_id,
                                'service': service_name,
                                'amount': total_amount,
                                'verified': verified,
                                'auto_linked': auto_link_success,
                                'is_renewal': is_renewal and bool(existing_username),
                                'old_expiry': old_expiry,
                                'new_expiry': new_expiry
                            })
                            
                            # Mark as processed
                            status = "RENEWAL_COMPLETED" if (is_renewal and existing_username) else "COMPLETED_AND_LINKED"
                            log_processed_id(PROCESSED_LOG, msgid, [], [bot_ref], status)
                            processed_ids.add(msgid)
                            break
                        else:
                            logging.error(f"ERROR: Failed to send notification for order #{order_id}")
                    else:
                        logging.error(f"ERROR: Failed to update database for order #{order_id}")
                else:
                    logging.error(f"ERROR: Panel account {'renewal' if (is_renewal and existing_username) else 'creation'} failed for order #{order_id}: {account_result.get('error')}")
                    continue
            
            # Mark email as seen
            mark_as_seen(mail, num)

        # Summary
        if bot_orders_processed:
            logging.info(f"SUCCESS: {len(bot_orders_processed)} accounts processed and delivered!")
            for order_info in bot_orders_processed:
                auto_linked = order_info.get('auto_linked', False)
                is_renewal = order_info.get('is_renewal', False)
                action = "RENEWED" if is_renewal else "CREATED"
                bot_ref = order_info.get('bot_ref', 'Unknown')
                
                if is_renewal:
                    logging.info(f"  Order #{order_info['order_id']} ({bot_ref}): {order_info['username']} (ID: {order_info['user_id']}) - {action} - Old: {order_info['old_expiry']} -> New: {order_info['new_expiry']} - Verified: {order_info['verified']} - Service: {order_info['service']}")
                else:
                    logging.info(f"  Order #{order_info['order_id']} ({bot_ref}): {order_info['username']} (ID: {order_info['user_id']}) - {action} - Verified: {order_info['verified']} - Auto-Linked: {auto_linked} - Service: {order_info['service']}")
        else:
            logging.info("No new bot orders processed.")

    except Exception as e:
        logging.exception("Exception during processing.")

    finally:
        try:
            mail.logout()
        except Exception:
            pass

if __name__ == "__main__":
    main()